/* ============ */
/* PokerTst.c	*/
/* ============ */
#include <defcodes.h>
#include <miscdefs.h>
#include <pokrdefs.h>
#include <mconf.h>
#include <math.h>

#define	NUM_PROBS	100
#define LOW_PROB	1.0e-4

/* ==================================================================== */
/* PokerTest - Runs Poker Test on random data roughly according to Knuth */
/* ==================================================================== */
void
main(void)
{
    int	    k;
    double  ChiSqProb[NUM_PROBS];
    double  DegFree, LoLimit, HiLimit;

    double  KnMinusProb, KnMinusStat, KnPlusProb, KnPlusStat;

    INIT_DATA_STRU  InitialData;
    POKER_DATA_STRU PokerData;

    AbortGracefully();			/* Make ^C act reasonably */

    printf("\tP O K E R  T E S T\n\n");
    GetInitialData(&InitialData);
    fflush(NULL); fprintf(stderr, "\n");

    /* -------------------------- */
    /* Print Initial Data Entries */
    /* -------------------------- */
    printf("Starting Seed = %u%s\n", InitialData.UserSeed,
	(InitialData.SeedSrce == (UINT)(-1)) ?
		" (Unsigned Integer Part of Time of Day)" : "");

    printf("Generator     = %s\n", InitialData.GenName);

    PokerData.RandFun = InitialData.RandFun;

    SetPokerControls(&PokerData);

    /* ---------------------------- */
    /* Initialize Count of Variates */
    /* ---------------------------- */
    PokerData.TotNumGen = 0;

    if (!PokerData.CallStatusOK)
    {
	int	HandCountOK;
	HandCountOK =
	    (PokerData.CardsPerHand - PokerData.NotEnough) >= MIN_CARDS;
	printf("\nOf the %d Possible Categories, ",
	    PokerData.CardsPerHand);

	if (PokerData.NotEnough == 1)
	{
	    printf("1 Has a Cell Expectation");
	}
	else
	{
	    printf("%d Have Cell Expectations", PokerData.NotEnough);
	}
	printf(" Less Than\nthe Number Requested (%d)",
	    PokerData.UserCellExpect);
	if (HandCountOK)
	{
	    printf(" and Will be Lumped With Others.\n");
	}
	else
	{
	    printf(".\n");
	}
	printf("For the Inputs That You Have Provided At Least %lu"
		" Hands are Required\nto Fill All Categories.\n",
		PokerData.UserNumHands);
	if (PokerData.UserCellExpect != MIN_CELL_XPCT)
	{
	    printf("For a Minimum of %d Samples Per Category At"
		" Least %lu Hands are Required.\n", MIN_CELL_XPCT,
		PokerData.IdealNumHands);
	}
	if (!HandCountOK)
	{
	    printf("At Least %d Cards Per Hand Are Required.\n",
		MIN_CARDS);
	    printf("Your Inputs Allow Only %d.\n",
		PokerData.CardsPerHand - PokerData.NotEnough);
	    exit(1);
	}
    }
    fflush(NULL);
    DegFree = PokerData.CardsPerHand - PokerData.NotEnough - 1;
    ChiSqDist(LOW_PROB, DegFree, &LoLimit, &HiLimit);

    P(printf("DegFree = %f, LoLimit = %.10e, HiLimit = %.10e\n",
	DegFree, LoLimit, HiLimit));
    /* ------------------------- */
    /* Generate Random Numbers,  */
    /* Calculate Chi-Square Data */
    /* ------------------------- */
    for (k = 0; k < NUM_PROBS; ++k)
    {
	CalcPokerChiSq(&PokerData);

	fprintf(stderr, "\rPass %3d (of %d), %8ld  Total Random Numbers",
	    k+1, NUM_PROBS, PokerData.TotNumGen);

	if (PokerData.PokerChiSq <= LoLimit)
	{
	    ChiSqProb[k] = LOW_PROB;
	}
	else if (PokerData.PokerChiSq >= HiLimit)
	{
	    ChiSqProb[k] = 1.0 - LOW_PROB;
	}
	else
	{
	    ChiSqProb[k] = chdtr(DegFree, PokerData.PokerChiSq);
	}
	if (ChiSqProb[k] < 0)
	{
	    fprintf(stderr, "\nChiSqFreq(): Function chdtr() "
		"Returned Negative Probability -  Can't Happen.\n");
	}
	P(printf("Pass %3d Poker Chi-Square Statistic = %f\n", k+1,
	   PokerData.PokerChiSq));
	P(printf("\t Chi-Square Probability on %.f"
		" Degrees of Freedom = %.4f\n", DegFree, ChiSqProb[k]));

    }

    /* -------------------------------------------------------- */
    /* Calculate K-S on Chi-Square Statistics and Probabilities */
    /* -------------------------------------------------------- */
    fflush(NULL);fprintf(stderr, "\n");
    KSCalc(ChiSqProb, NUM_PROBS,
		&KnPlusStat, &KnPlusProb,
 		&KnMinusStat, &KnMinusProb);

   printf("\nKolmogorov-Smirnov Statistics and Probabilities"
	" on Chi-Square Data\n");

   printf("\tK(%d)+ = %f (Knuth) or %9f%%\n", NUM_PROBS,
	sqrt((double)NUM_PROBS)*KnPlusStat, 100*KnPlusProb);

   printf("\tK(%d)- = %f (Knuth) or %9f%%\n", NUM_PROBS,
	sqrt((double)NUM_PROBS)*KnMinusStat, 100*KnMinusProb);

    printf("\nThis Run Required %ld Random Numbers (%s)\n\n",
	PokerData.TotNumGen, InitialData.GenName);
}
